package display;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JButton;
import javax.swing.JPanel;

/**
 * @author Fabien Dagnat <fabien.dagnat@imt-atlantique.fr>
 */
public class DisplayedThread extends Thread {
	private boolean suspended = true;
	private int angle = 0;
	private final int rate;
	public final static int STEP = 6;
	private final Runnable target;
	private final GraphicCanvas display;
	private final JPanel panel;

	public DisplayedThread(Runnable target, int rate, String title) {
		this(target, rate, title, Config.lifeColor);
	}

	public DisplayedThread(Runnable target, int rate, String title, Color c) {
		this.target = target;
		this.rate = rate;
		this.display = new GraphicCanvas(title, c);
		this.display.setColor(Config.suspColor);
		this.panel = createPanel();
	}

	public JPanel getPanel() {
		return this.panel;
	}

	@Override
	public void run() {
		try {
			mysuspend();
			this.target.run();
			this.display.setColor(Color.white);
			this.display.setAngle(0);
		} catch (InterruptedException e) {}
	}

	private JPanel createPanel() {
		JPanel result = new JPanel();
		// Set up Buttons
		result.setFont(Config.buttonFont);
		Dimension dimButton = new Dimension(70, 40);
		JPanel p = new JPanel();
		JButton run = new JButton("Run");
		run.setPreferredSize(dimButton);
		run.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				DisplayedThread.this.activate();
			}
		});
		JButton pause = new JButton("Pause");
		pause.setPreferredSize(dimButton);
		pause.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				DisplayedThread.this.passivate();
			}
		});
		p.add(run);
		p.add(pause);
		result.setLayout(new BorderLayout());
		result.add(BorderLayout.SOUTH, p);
		result.add(BorderLayout.NORTH, this.display);
		return result;
	}

	private synchronized void mysuspend() throws InterruptedException {
		while (this.suspended)
			wait();
	}

	void activate() {
		if (this.suspended) {
			this.suspended = false;
			this.display.setColor(Config.actColor);
			synchronized (this) {
				notify();
			}
		}
	}

	void passivate() {
		if (!this.suspended) {
			this.suspended = true;
			this.display.setColor(Config.suspColor);
		}
	}

	public static void rotate(int degrees) {
		DisplayedThread d = (DisplayedThread) Thread.currentThread();
		for (int i = 0; i < degrees; i += STEP)
			synchronized (d) {
				try {
					d.mysuspend();
					d.angle = (d.angle + STEP) % 360;
					d.display.setAngle(d.angle);
					Thread.sleep(d.rate);
				} catch (InterruptedException e) {}
			}
	}
}
